/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include "typedefs.h"                    // Required
#include "usb.h"                         // Required
#include "io_cfg.h"                      // Required
#include <math.h>

#define VER_HIGH	5
#define VER_LOW		29
// recaptured version.

/** C O N F I G U R A T I O N ************************************************/

#if   defined(__18F4550)||defined(__18F4455)|| \
	  defined(__18F2550)||defined(__18F2455)|| \
      defined(__18F4553)||defined(__18F4458)|| \
      defined(__18F2553)||defined(__18F2458)

#pragma config PLLDIV   = 5       		// (20 MHz input)
#pragma config CPUDIV   = OSC1_PLL2		//
#pragma config USBDIV   = 2       		// Clock source from 96MHz PLL/2
#pragma config FOSC     = HSPLL_HS		//
#pragma config FCMEN    = OFF			//
#pragma config IESO     = OFF
#pragma config PWRT     = OFF
#pragma config BOR      = ON
#pragma config BORV     = 3
#pragma config VREGEN   = ON
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = ON
#pragma config LPT1OSC  = OFF			// you could enable this if you want.
#pragma config PBADEN   = OFF
#pragma config CCP2MX   = OFF
#pragma config STVREN   = ON			// Stack
#pragma config LVP      = ON
#pragma config ICPRT    = OFF       	// Dedicated In-Circuit Debug/Programming
#pragma config XINST    = OFF       	// Extended Instruction Set
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CP2      = OFF
#pragma config CP3      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRT2     = OFF
#pragma config WRT3     = OFF
#pragma config WRTB     = ON      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTR2    = OFF
#pragma config EBTR3    = OFF
#pragma config EBTRB    = OFF
//
#endif
/** P R I V A T E  P R O T O T Y P E S ***************************************/
#define VERSION_DELAY					1500			// in ms
#define PERSISTENT_SETTINGS_ENABLE		1				// 1= use EEPROM to save settings, 0=disable persistent settings
#define T2_PERIOD_DEFAULT 				82
#define MAX_PWM_DUTY_LOW_POWER			50
#define MIN_PWM_DUTY					1
#define USB_CONNECT_TIMEOUT				10				// 20 minutes when on USB power
#define USB_CONNECT_TIMEOUT_BATTERY		2				// 15 seconds when on battery
#define DEFAULT_TIMEOUT					5				// 5 seconds for backlight
#define DEFAULT_DISPLAY_TIMEOUT			30				// 30 seconds for reverting back to display default mode
#define DEFAULT_AUTO_BACKLIGHT_MODE		1				// from 6pm to 6am
#define AUTO_NIGHT_HOUR					18				// 6 pm
#define AUTO_MORNING_HOUR				6				// 6 am
#define AUTO_BACKLIGHT_TIMEOUT			3				// 3 seconds after auto backlight disabled, the backlight will start dimming.
#define LV_DIMMING_FACTOR				0.9				// reduce PWM Duty by 20% when low power 
#define NOBATTERY_DEFAULT				1				// (0) Battery Installed (1) No Battery operation as default
#define TWENTYFOURHOUR_DEFAULT			0				// (0) 24 hour time as default (1) 12 hour time as default
//
#define LOW_POWER_PERIOD				5				// prefer odd numbers
#define SYNC_DISPLAY_PERIOD				9				
#define LO_DISPLAY_PERIOD				7
#define LOW_POWER_LV_TRIP				3600			// in mV
#define EXTRA_LOW_POWER_LV_TRIP			3000			// trip voltage point for extra low power conservation mode in mV 
#define MAX_HYSTERISIS					1000			// in mV
// Note: EXTRA_LOW_POWER_LV_TRIP must be lower than LOW_POWER_LV_TRIP
#define HYSTERISIS_DEFAULT				250				// in mV
//
#define MAGIC_VALUE						76				// this can be any byte value except not 0xFF!
#define BUTTON_DELAY					40
#define EEPROM_WRITTEN_BASE_ADDR		0
#define	CLOCKMODE_ADDR					1
#define LOWPOWERPWMDUTY_ADDR			2
#define TIMEOUTPERIODHIGH_ADDR			3
#define TIMEOUTPERIODLOW_ADDR			4
#define LVTRIPPOINTHIGH_ADDR			5
#define LVTRIPPOINTLOW_ADDR				6
#define USBREGASSUMEDVOLTAGEHIGH_ADDR	7
#define USBREGASSUMEDVOLTAGELOW_ADDR	8
#define SENSERESISTORHIGH_ADDR			9
#define SENSERESISTORLOW_ADDR			10
#define SCROLLINGMODE_ADDR				11
#define DISPLAYTIMEOUTHIGH_ADDR			12
#define DISPLAYTIMEOUTLOW_ADDR			13
#define AUTOBACKLIGHTMODE_ADDR			14
#define TWENTYFOURHOURMODE_ADDR			15
#define NOBATTERY_ADDR					16
#define HYSTERISISHI_ADDR				17
#define HYSTERISISLO_ADDR				18
//
#define MODE_DISPLAY_TIME_HH_MM            0    //Display Time HH:MM mode
#define MODE_DISPLAY_TIME_MM_SS            1    //Display Time MM:SS mode
#define MODE_DISPLAY_DATE_DD_MM            2    //Display Date DD:MM mode
#define MODE_DISPLAY_YEAR_YYYY             3    //Display Year YYYY mode
#define MODE_DISPLAY_CHARGING_CURRENT_MA   4    //Display Charging Current in mA mode
#define MODE_DISPLAY_SUPPLY_VOLTAGE        5    //Display Supply Voltage in mV mode
#define MODE_DISPLAY_CHARGE_STATE          6	//Display Charge State of the Battery
#define MODE_DISPLAY_PWM_DUTY_CYCLE        7    //Display Backlight PWM Duty Cycle mode
#define MODE_DISPLAY_USB_STATE			   8    //Display USB connection status
#define MODE_DISPLAY_ALL                   9	//Display Extended Scrolling Time Mode
#define MODE_DISPLAY_TIME_SCROLLING        10	//Display Scrolling Time Mode
#define MODE_DISPLAY_VERSION			   11	//Display the Firmware version
#define MODE_MODULUS					   12	//Total Number
//
#define DEFAULT_DISPLAY_MODE			(MODE_DISPLAY_TIME_MM_SS)
#define DEFAULT_DEFAULT_DISPLAY_MODE    (DEFAULT_DISPLAY_MODE)
#define NUMBER_OF_MONTHS				12
#define YEAR_OFFSET						1900
#define MIN_YEAR						(1900-YEAR_OFFSET)
#define MAX_YEAR						(2599-YEAR_OFFSET)
#define SENSE_RESISTOR_DEFAULT			1650			// in mOhms corresponds to the default 2 x 3.3 Ohm resistors in parallel.
#define VOLTAGE_REF_DEFAULT				3300			// in mV
//
#define BATTERY_LOWEST_VOLTAGE			3000			// in mV NiMh can go as low as 1.1V per cell so assume 1.0 then
#define BATTERY_HIGHEST_VOLTAGE			4800			// in mV NiMh can go as high as 1.6 per cell (4.8=1.6*3)
#define BASIC_USB_STATE					(POWERED_STATE)	// fall back for USB enumeration system.
//
/** V E C T O R  R E M A P P I N G *******************************************/
int systemTimeOut(void);
void setSystemTimeOut(int);
void displayVersion(void);
void setTimeOut(int);
void myISR(void);
void myISRHigh(void);
void disAdec(int, int);
void disA(int, int);
void disWdec(int);
void updateANs(void);
unsigned int readAN(unsigned char);
void setPWMDuty(int);
void hitBackLightPWMDuty(int, int);
void USBProtocolResetHandler(void);
void setClockMode(char);
void resetDisplayTimeOut(void);
int WriteEEPROM(int, int);
int ReadEEPROM(int);
void prepareInfoPacket(void);
//void getInfoPacket(void);
void restoreAllSettings(void);
void saveAllSettings(void);
void updateAllSettings(void);
void setAutoBackLightMode(int);
int initHLVD(unsigned char, int);
//
#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}

#pragma code
/** D E C L A R A T I O N S **************************************************/
#pragma code

void
DelayMs(int delay)
{
	unsigned int n;

	while(delay--) {
		for(n = 0; n < 500; n++) {
		// the following must be around a 2us delay...	
			_asm
				nop
				nop
				nop
				nop
			_endasm
		}
	}
}

rom char sevenSegment[16]={ 0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7D, 0x07, 0x7F, 0x6F, 0x77, 0x7C, 0x39, 0x5E, 0x79, 0x71 };

//rom int CenturiesTable[NUMBER_OF_CENTURIES]=
//						{ 	0, 		// 1900-1999
//							6,		// 2000-2099
//							4,		// 2100-2199
//							2,		// 2200-2299
//							0,		// 2300-2399
//							6,		// 2400-2499
//							4		// 2500-2599
//						  };
//
//rom int MonthsTable[NUMBER_OF_MONTHS]=
//						{ 	0,			// Jan
//							3,			// Feb
//							3,			// Mar
//							6,			// Apr
//							1,			// May
//							4,			// Jun
//							6,			// Jul
//							2,			// Aug
//							5,			// Sep
//							0,			// Oct
//							3,			// Nov
//							5			// Dec
//						};

rom int DaysOfTheMonth[NUMBER_OF_MONTHS]={ 	
	31,			// Jan
	28,			// Feb
	31,			// Mar
	30,			// Apr
	31,			// May
	30,			// Jun
	31,			// Jul
	31,			// Aug
	30,			// Sep
	31,			// Oct
	30,			// Nov
	31			// Dec
};
/** V A R I A B L E S ********************************************************/
#pragma udata
//
char usbConnecting;						// =1 when USBConnect is active. 0 otherwise
char syncNeeded;						// !=0 when a sync is needed			
char flagchange;
char decimalPoint;						// 1=COL segment 2=DCP3 4=DCP2 8=DCP1
char digits[5];							// screen digits[4] used for scrolling modes
char twentyfourhourmode;					
int screenCounter;
int screenPeriod;
char screenUpdateOn;
int  hlvdset;
int hlvdreq;
int actualtrippoint;
int buttonpressedlast;
int buttonpressednow;
char buttonkey;
int usbRegAssumedVoltage;
int batteryvoltage;
int displayCountDown;
// **** Actual USB Clock Global Variables Communicated ******
int supplyVoltage;
int lastGoodSupplyVoltage;
int currentChargeRate;
int senseresistor;
int an0, an1;
int RTCcountdown, RTCsystemcountdown;
int RTCseconds;
int displayCount;
int syncCount;
int loCount;
int extralowCount;
int RTCminutes;
int RTChours;
int RTCwday;
int RTCmday;
int RTCyday;
int RTCisdst;
int RTCmonth;
int RTCyear;
char verhigh;
char verlow;
int lowPowerPWMDuty;
int timeOutPeriod;
int lvtripPoint;				// in mV
int lastseconds;
int lastminutes;
int lasthours;
int lastwday;
int lastyday;
int lastmday;
int lastisdst;
int lastmonth;
int lastyear;
int hysterisis;
char clockmode;					// current display mode
char requestReset;				// bit 0=1 when a hard reset has been requested with settings going back to defaults! bit1=1 when a soft request has been requested! (bit 1 takes precedence)
char powermode;					// bit 0=0 means High Power bit 0=1 Low Power (<1mA) bit1=1 Very Low Power bit2=1 Extra Low Power mode
//char reqPowermode;				// request power mode value
char powersense;				// =0 means the USB cable is disconnected (running from batteries), =1 means USB cable is connected.
char noBattery;					// =0 for normal operation, =1 if not using a battery.
char scrollingmode;				// the default display mode
char autobacklightmode;			// 0= off 1=between 6pm and 6am =2 always on (except when on battery power).
int displaytimeout;
int screenRefreshes;
int currentPWMDuty;
char lowPowerModeEnable;
//
void USBConnect(void);
//
#pragma code
//
// powermode =0 high power mode
// powermode =1 low power mode
// powermode =2 high power below low voltage trip point
// powermode =3 low power below low voltage trip point
// powermode =4 below extra low voltage point, so extra low power mode
void goToLowPower(void)
{
	if((powermode & 0x04)==0)
	{
		// when bit 2 (powermode)=0 then...
	if((lowPowerModeEnable==1)&&(currentPWMDuty==0)&&(usbConnecting==0))
	{
		powermode=(powermode|0x01);
		OSCCONbits.IDLEN=1;
		OSCCONbits.SCS1=1;
		OSCCONbits.SCS0=1;
		Sleep();
	} 
	else 
	 {
		powermode=powermode & 0xFE;
		if(usb_device_state==CONFIGURED_STATE)lowPowerModeEnable=1;
	 }
 	} else
	{
			// extra low power sleep!
			PIE2=0;
			PIE1=0;
			UIR=0;
			PIR1=0;
			PIR2=0;
			UCON=0;
			PIE1bits.TMR1IE=1;
			INTCONbits.GIEL=0;			// turn off low interrupts
			OSCCONbits.IDLEN=0;
			Sleep();					// go to sleep!
	}
}

void serveKeyPress(int x)
{
	int k;
	buttonpressednow=screenCounter;
	k=buttonpressednow-buttonpressedlast;
	if(k<0)k=-k;
	if(k>x)
	{
		buttonpressedlast=buttonpressednow;
		if(buttonkey==0)buttonkey=1;
	} else
	{
		// do nothing...
	}
	// serve the key press
	if(buttonkey==0)
		{
				// button not pressed					
		} else 
		if(buttonkey==1)
		{ 
				// button pressed
				resetDisplayTimeOut();
				if(systemTimeOut())
				{
					hitBackLightPWMDuty(lowPowerPWMDuty, timeOutPeriod);
					setSystemTimeOut(2);
				} else
				{
					setClockMode(clockmode+1);
					resetDisplayTimeOut();
				}
				buttonkey=0;
		}
}

void refreshScreen(void)
{
	int i, j;
	int a0, b0, c0, d0, e0;				// for the LCD pins
	int a1, b1, c1, d1, e1;				// for the latches

	screenRefreshes++;
	screenCounter++;
	b0=0;
	b1=0;
	if(PORTBbits.RB6==1){ b0|=0x40; b1|=0x40; }
	else 
	{ 
		// button pressed
		b0&=~0x40; b1&=~0x40; 
	}
	// build up a0 and a1 outputs
	a0=0;
	a1=0;
	if((digits[0] & 0x02)!=0)a0|=0x04;			// RA2 = 1B
	if((digits[0] & 0x01)!=0)a0|=0x10;			// RA4 = 1A
	if((digits[0] & 0x20)!=0)a0|=0x20; 			// RA5 = 1F
	//
	if((decimalPoint & 0x08)!=0)a1|=0x04;		// latched RA2 = DP1
	if((digits[0] & 0x04)!=0)a1|=0x10; 			// latched RA4 = 1C
	if((digits[0] & 0x08)!=0)a1|=0x20; 			// latched RA5 = 1D
	//

	// build up b0 and b1 outputs
	if((decimalPoint & 0x01)!=0)b0|=0x01;		// RB0 = Col
	if((digits[1]& 0x02)!=0)b0|=0x02;			// RB1 = 2B
	if((digits[1]& 0x01)!=0)b0|=0x04;			// RB2 = 2A
	if((digits[1]& 0x20)!=0)b0|=0x08;			// RB3 = 2F
	if((digits[1]& 0x40)!=0)b0|=0x10;			// RB4 = 2G
	//
	if((digits[2] & 0x10)!=0)b1|=0x01;			// latched RB0 = 3E
	if((decimalPoint & 0x04)!=0)b1|=0x02;		// latched RB1 = DP2
	if((digits[1]& 0x04)!=0)b1|=0x04;			// latched RB2 = 2C
	if((digits[1]& 0x08)!=0)b1|=0x08;			// latched RB3 = 2D
	if((digits[1]& 0x10)!=0)b1|=0x10;			// latched RB4 = 2E
	//

	// build up PORTC output
	c0=0;
	c1=0;
	if((digits[3] & 0x20)!=0)c0|=0x40;			// RC6 = 4F
	if((digits[3] & 0x40)!=0)c0|=0x80;			// RC7 = 4G
	//
	if((digits[3] & 0x04)!=0)c1|=0x40;			// latched RC6 = 4C
	if((digits[3] & 0x08)!=0)c1|=0x80;			// latched RC7 = 4D
	//

	// build up PORTD output
	d0=0;
	d1=0;
	if((digits[3] & 0x01)!=0)d0|=0x04;			// RD2 = 4A
	if((digits[2] & 0x02)!=0)d0|=0x10;			// RD4 = 3B
	if((digits[2] & 0x01)!=0)d0|=0x20;			// RD5 = 3A
	if((digits[2] & 0x20)!=0)d0|=0x40;			// RD6 = 3F
	if((digits[2] & 0x40)!=0)d0|=0x80;			// RD7 = 3G
	//
	if((digits[3] & 0x02)!=0)d1|=0x04;			// latched RD2 = 4B
	if((digits[3] & 0x10)!=0)d1|=0x10;			// latched RD4 = 4E
	if((decimalPoint & 0x02)!=0)d1|=0x20;		// latched RD5 = DP3
	if((digits[2] & 0x04)!=0)d1|=0x40;			// latched RD6 = 3C
	if((digits[2] & 0x08)!=0)d1|=0x80;			// latched RD7 = 3D

	// build up PORTE output
	e0=0;
	e1=0;
	if((digits[0] & 0x40)!=0)e0|=0x01;			// RE0 = 1G
	//
	if((digits[0] & 0x10)!=0)e1|=0x01;			// latched RE0 = 1E
	//
	if((screenCounter & 0x01)!=0)
	{ 
				i=0xFF;
	} 
	else 
	{
				i=0x00;
	}
	a0= a0 ^ i;
	a1= a1 ^ i;
	b0= b0 ^ i;
	b1= b1 ^ i;
	c0= c0 ^ i;
	c1= c1 ^ i;
	d0= d0 ^ i;
	d1= d1 ^ i;
	e0= e0 ^ i;
	e1= e1 ^ i;
	
	if(PORTCbits.RC2==1){ c0|=0x04; c1|=0x04; } else { c0&=~0x04; c1&=~0x04; }
	
	e1 = e1 | 0x02;					// RE1 = LE of IC3
	d1 = d1 | 0x08;					// RD3 = LE of IC2
	e0 = e0 & (~0x02);				// RE1 = LE of IC3
	d0 = d0 & (~0x08);				// RD3 = LE of IC2

	PORTA=a1;
	PORTB=b1;
	PORTC=c1;
	PORTD=d1;
	PORTE=e1;
		_asm
			nop
			nop
			nop
			nop
		_endasm
	PORTEbits.RE1=0;
	PORTDbits.RD3=0;
		_asm
			nop
			nop
			nop
			nop
		_endasm
	PORTE=e0;
	PORTD=d0;
	PORTC=c0;
	PORTB=b0;
	PORTA=a0;
}

void doHLVDInterrupt(void)
{
	hlvdset++;
	if(currentPWMDuty>0)
	{
		setPWMDuty(currentPWMDuty*LV_DIMMING_FACTOR);
	}
	else
	{
		// so the backlight is off but still the HLVD interrupt was flagged, so we must go even lower now!
		if((powermode & 0x02)==0)
		{
			// so we switch to extra low power conservation mode...
			powermode=0;
			actualtrippoint=initHLVD(0, EXTRA_LOW_POWER_LV_TRIP);		// note initHLVD sets the powermode to zero always...
			powermode|=0x02;	
			loCount=0;
		}   
		else
		{
			if((powermode & 0x04)==0)
			{
			saveAllSettings();
			powermode|=0x04;
			extralowCount=0;
			}
		}
	}
}

int timeOut(void)
{
	if(RTCcountdown>0)return FALSE; else return TRUE;
}

void setTimeOut(int seconds)
{
	RTCcountdown=seconds;
}

int systemTimeOut(void)
{
	if(RTCsystemcountdown>0)return FALSE; else return TRUE;
}

void setSystemTimeOut(int seconds)
{
	RTCsystemcountdown=seconds;
}

int isLeapYear(int iyear)
{
	// returns TRUE if iyear is a leap year, FALSE otherwise
	int i, j;
	i=(iyear%4);
	if(i!=0)return 0;	// since not divisible by 4
	i=(iyear%100);
	if(i!=0)return 1;
	j=(iyear%400);
	if(j==0)return 1; else return 0;
}

void RTCUpdater(void)
{
	int i, j;

	RTCseconds++;
	if(RTCcountdown>0)RTCcountdown--; else RTCcountdown=0;
	if(RTCsystemcountdown>0)RTCsystemcountdown--; else RTCsystemcountdown=0;
	if(displayCountDown>0)displayCountDown--; else 
		{
		// a displayTimeOut has occured!
		displayCountDown=0;
		setClockMode(scrollingmode);
		}							
	if(RTCseconds>59)
	{ 	
	RTCseconds=0; 
	RTCminutes++; 
	}
	if(RTCminutes>59)
	{
	RTChours++;
	RTCminutes=0;
	}
	if(RTChours>23)
	{
	RTChours=0;
	RTCwday=((RTCwday +1)%7);
	RTCmday++;
	RTCyday++;
	i=isLeapYear(YEAR_OFFSET+RTCyear);
	j=DaysOfTheMonth[RTCmonth];
	if((RTCmonth==1)&&(i==1))j++;				// a leap year for Feb
	if(RTCmday>j)
				{
				RTCmday=1;
				RTCmonth++;	
				if(RTCmonth>=NUMBER_OF_MONTHS)
				{
				RTCmonth=0;
				RTCyear++;
				RTCyday=0;
				}
		}	
	}
}

	
int getBatteryChargePercent(void)
{
	float f;
	int batterycharge;
	if(an0>0)supplyVoltage=(int)((float)usbRegAssumedVoltage*1023.0/(float)an0); else supplyVoltage=0.0;
	f=(float)supplyVoltage*(float)an1/1023.0;
	batteryvoltage=(supplyVoltage-f);
	// an1=0 usb_device_sate> BASIC STATE implies that connected by USB cable no battery
	// an1=0 usb_device_state<=BASIC_STATE implies that connected by battery no USB
	// an1>0 means battery and USB power.
	if((an1>0)||((an1==0)&&(usb_device_state<=BASIC_USB_STATE)))
	{
		// if battery and no USB or battery and USB
		batterycharge=100*(((float)batteryvoltage-BATTERY_LOWEST_VOLTAGE)/(lastGoodSupplyVoltage-BATTERY_LOWEST_VOLTAGE));
	} else batterycharge=0;
	if(batterycharge<0)batterycharge=0; else if(batterycharge>100)batterycharge=100;
	return batterycharge;
}

void displayClockMode(int iclockmode)
{
		int i, v1, j;
		float f;
		syncCount=(syncCount % SYNC_DISPLAY_PERIOD);
		if((syncCount==0)&&((syncNeeded==1)&&((clockmode==MODE_DISPLAY_TIME_HH_MM)||(clockmode==MODE_DISPLAY_TIME_MM_SS)||(clockmode==MODE_DISPLAY_DATE_DD_MM)||(clockmode==MODE_DISPLAY_YEAR_YYYY)
		||(clockmode==MODE_DISPLAY_TIME_SCROLLING)||(clockmode==MODE_DISPLAY_ALL))))
		{
			digits[0]=0x6D;		// S
			digits[1]=0x6E;		// Y
			digits[2]=0x54;		// N
			digits[3]=0x58;		// C
			decimalPoint=0;
			displayCount=0;
		} else
		if(iclockmode==MODE_DISPLAY_TIME_HH_MM)
		{
			if(twentyfourhourmode==0)disAdec(RTChours, 0); 
			else 
			{
				if(RTChours<=12)disAdec(RTChours,0);
				else disAdec(RTChours-12,0);
			}				
			disAdec(RTCminutes,2);
			if((RTCseconds & 0x02)==0)decimalPoint=0; else decimalPoint=1;	
		} else
		if(iclockmode==MODE_DISPLAY_TIME_MM_SS)
		{
			disAdec(RTCminutes,0);
			disAdec(RTCseconds,2);
			if((RTCseconds & 0x01)!=0)decimalPoint=0; else decimalPoint=1;
		} else
		if(iclockmode==MODE_DISPLAY_DATE_DD_MM)
		{
			// display date mode
			displayCount=displayCount % 4;
			i=displayCount;
			if(i==0)
			{
			digits[0]=0x5E;		// d
			digits[1]=0x77;		// A
			digits[2]=0x6E;		// Y			
			digits[3]=0;
			decimalPoint=0;
			} else
			{		
			disAdec(RTCmday,0);
			disAdec(RTCmonth+1,2);
			decimalPoint=4;
			}
			displayCount++;

		} else
		if(iclockmode==MODE_DISPLAY_YEAR_YYYY)
		{
			// display the year
			displayCount=displayCount % 4;
			i=displayCount;
			if(i==0)
			{
			digits[0]=0x6E;		// Y
			digits[1]=0x79;		// E
			digits[2]=0x77;		// A
			digits[3]=0x50;		// R
			decimalPoint=0;
			} else
			{
			disWdec(RTCyear+YEAR_OFFSET);
			decimalPoint=0;
			}
			displayCount++;
		} else
		if(iclockmode==MODE_DISPLAY_CHARGING_CURRENT_MA)
		{
			if(an0>0)f=usbRegAssumedVoltage*1023.0/(float)an0; else f=0.0;	// f in mV is the supply voltage
			f=f*an1/1023.0;
			if(senseresistor>0)f=(1000.0*f/(float)senseresistor); else f=0.0;
			currentChargeRate=(int)f;
			disWdec(currentChargeRate);
			//disWdec(powermode);
			digits[0]=0x39;		// C
			decimalPoint=0x08;
		} else
		if(iclockmode==MODE_DISPLAY_VERSION)
		{
			displayVersion();
		} else
		if(iclockmode==MODE_DISPLAY_CHARGE_STATE)
		{
			v1=getBatteryChargePercent();
			if(v1==0)
			{
				digits[1]=0x40;
				digits[2]=0x40;
				digits[3]=0x40;
			} else
			{
				disWdec(v1);
			}
			digits[0]=0x7C;		// b
			decimalPoint=0;
		} else
		if(iclockmode==MODE_DISPLAY_SUPPLY_VOLTAGE)
		{
			if(an0>0)supplyVoltage=(int)((float)usbRegAssumedVoltage*1023.0/(float)an0); else supplyVoltage=0.0;
			disWdec(supplyVoltage/10.0);
			digits[0]=0x7C;		// b
			decimalPoint=4;
		} else
		if(iclockmode==MODE_DISPLAY_PWM_DUTY_CYCLE)
		{
			if(currentPWMDuty!=0)
			{
			disWdec(currentPWMDuty);
			decimalPoint=0;
			} else
			{
			disAdec(powermode,2);
			digits[1]=0x50;		// r
			decimalPoint=1;
			}
			digits[0]=0x73;		// P
		} else
		if(iclockmode==MODE_DISPLAY_USB_STATE)
		{
			digits[0]=0x3E;		// U
			digits[1]=0x6D;		// S
			digits[2]=0x7C;		// B
			digits[3]=sevenSegment[usb_device_state & 0x0F];	
			decimalPoint=0;
		} else
		if(iclockmode==MODE_DISPLAY_ALL)
		{
			displayCount=displayCount % 28;
			i=displayCount;
			if(i==0)
			{
				digits[0]=0;
				digits[1]=0;
				digits[2]=0;
				digits[3]=0;
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==1)
			{
				digits[4]=sevenSegment[ (RTChours/10) & 0x0F];
				decimalPoint=0;
			} else
			if(i==2)
			{
				digits[4]=sevenSegment[ (RTChours % 10)& 0x0F];
				decimalPoint=0;
			} 
			else
			if(i==3)
			{
				digits[4]=sevenSegment[ (RTCminutes /10) & 0x0F];
				decimalPoint=2;
			}
			else
			if(i==4)
			{
				digits[4]=sevenSegment[ (RTCminutes % 10) & 0x0F];
				decimalPoint=4;
			}
			else
			if(i==5)
			{
				digits[4]=0;
				decimalPoint=8;
			} else
			if(i==6)
			{
				digits[4]=0x5E;		// d
				decimalPoint=0;
			} else
			if(i==7)
			{
				digits[4]=0x77;		// A
				decimalPoint=0;
			} else
			if(i==8)
			{
				digits[4]=0x6E;		// Y			
				decimalPoint=0;
			} else
			if(i==9)
			{
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==10)
			{
				digits[4]=sevenSegment[ (RTCmday/10) & 0x0F];
				decimalPoint=0;
			} else
			if(i==11)
			{
				digits[4]=sevenSegment[ (RTCmday % 10) & 0x0F];
				decimalPoint=0;
			} else
			if(i==12)
			{
				digits[4]=sevenSegment[ ((RTCmonth+1)/10) & 0x0F];
				decimalPoint=2;
			}
			else
			if(i==13)
			{
				digits[4]=sevenSegment[ ((RTCmonth+1)%10) & 0x0F];
				decimalPoint=4;
			}
			else
			if(i==14)
			{
				digits[4]=0;
				decimalPoint=8;
			} else
			if(i==15)
			{
				digits[4]=0x6E;		// Y
				decimalPoint=0;
			} else
			if(i==16)
			{
				digits[4]=0x79;		// E
				decimalPoint=0;
			} else
			if(i==17)
			{
				digits[4]=0x77;		// A
				decimalPoint=0;	
			} else
			if(i==18)
			{
				digits[4]=0x50;		// r
				decimalPoint=0;
			} else
			if(i==19)
			{		
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==20)
			{
				j=(RTCyear+YEAR_OFFSET);
				digits[4]=sevenSegment[ (j/1000) & 0x0F];
				decimalPoint=0;
			}
			else
			if(i==21)
			{
				j=(RTCyear+YEAR_OFFSET);
				j=j%1000;
				digits[4]=sevenSegment[ (j/100) & 0x0F];
				decimalPoint=0;
			}
			else
			if(i==22)
			{
				j=(RTCyear+YEAR_OFFSET);
				j=j % 100;
				digits[4]=sevenSegment[ (j/10) & 0x0F];
				decimalPoint=0;
			} else
			if(i==23)
			{
				j=(RTCyear+YEAR_OFFSET);
				j=j % 10;
				digits[4]=sevenSegment[ (j & 0x0F)];
				decimalPoint=0;
			} else
			if(i==24)
			{
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==25)
			{
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==26)
			{
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==27)
			{
				digits[4]=0;
				decimalPoint=0;
			} 
			digits[0]=digits[1];
			digits[1]=digits[2];
			digits[2]=digits[3];
			digits[3]=digits[4];			// shift one place
			displayCount++;
		} else
		if(iclockmode==MODE_DISPLAY_TIME_SCROLLING)
		{
			displayCount=displayCount % 10;
			i=displayCount;
			if(i==0)
			{
				digits[0]=0;
				digits[1]=0;
				digits[2]=0;
				digits[3]=0;
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==1)
			{
				digits[4]=sevenSegment[ (RTChours/10) & 0x0F];
				decimalPoint=0;
			} else
			if(i==2)
			{
				digits[4]=sevenSegment[ (RTChours % 10)& 0x0F];
				decimalPoint=0;
			} 
			else
			if(i==3)
			{
				digits[4]=sevenSegment[ (RTCminutes /10) & 0x0F];
				decimalPoint=2;
			}
			else
			if(i==4)
			{
				digits[4]=sevenSegment[ (RTCminutes % 10) & 0x0F];
				decimalPoint=4;
			}
			else
			if(i==5)
			{
				digits[4]=sevenSegment[ (RTCseconds /10) & 0x0F];
				decimalPoint=10;
			} 
			else
			if(i==6)
			{
				digits[4]=sevenSegment[ (RTCseconds % 10) & 0x0F];
				decimalPoint=4;
			}
			else
			if(i==7)
			{
				digits[4]=0;
				decimalPoint=8;
			} else
			if(i==8)
			{
				digits[4]=0;
				decimalPoint=0;
			} else
			if(i==9)
			{
				digits[4]=0;
				decimalPoint=0;
			}
			digits[0]=digits[1];
			digits[1]=digits[2];
			digits[2]=digits[3];
			digits[3]=digits[4];			// shift one place
			displayCount++;
			}
			else
			{
			digits[0]=0x40;
			digits[1]=0x40;
			digits[2]=0x40;
			digits[3]=0x40;
			decimalPoint=0;
			}
	syncCount++;	
}

void ScreenUpdater(void)
{
		if((powermode & 0x02)!=0)
		{
			loCount=(loCount % LO_DISPLAY_PERIOD);
			if(an0>0)supplyVoltage=(int)((float)usbRegAssumedVoltage*1023.0/(float)an0); else supplyVoltage=0;
			if(supplyVoltage>(lvtripPoint+hysterisis))
			{
				// we must exit the low voltage trip point here...
				powermode=0;
				actualtrippoint=initHLVD(0, lvtripPoint);			// put it back up! note initHLVD sets powermode to zero
				displayClockMode(clockmode);
			} else
			{
			if(loCount==0)
			{
			digits[0]=0x00;		// 
			digits[1]=0x38;		// L
			digits[2]=0x5C;		// o
			digits[3]=0x00;		// 
			decimalPoint=0;
			displayCount=0;
			syncCount=0;
			} else
			{
				displayClockMode(clockmode);
			}
			loCount++;
		 }
		} else
		{
			   displayClockMode(clockmode);
		}
}

void checkAutoBackLight(void)
{
	//decimalPoint=0x0F;
	if((((autobacklightmode & 0x03)==0x02)||(((autobacklightmode & 0x03)==1)&&((RTChours>=AUTO_NIGHT_HOUR)||(RTChours<AUTO_MORNING_HOUR))))&&(powersense==1))
	{
		hitBackLightPWMDuty(lowPowerPWMDuty, AUTO_BACKLIGHT_TIMEOUT);
		autobacklightmode|=0x80; 				// auto on indicator!
	} else
	{
		// do nothing, it will dim after the timeout period.
		autobacklightmode&=0x7F;
	}
}

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	OSCCONbits.SCS1=0;
	OSCCONbits.SCS0=0;			// note: SCS1 should be cleared first!
	TMR1H+=0x80;	
	RTCUpdater();
	updateANs();
	if((powermode & 0x04)==0)
	{
		if(screenUpdateOn!=0)
		{
		ScreenUpdater();
		} 			
		// now update the state of powersense 0=battery 1=USB power: powersense is thus updated @ 1Hz.
		if((noBattery==1)||(an1!=0))
		{
			powersense=1;			// if no battery mode is enabled then always assume USB power, otherwise if charging current!=0
		} 
		else
		{
			if(powersense==1)
			{
			// here we have detected a USB cable detachment, ie. we have gone from powersense=1 to powersense=0
			// ie. being USB powered to now under battery power (or battery fully charged?)... Must reset the USB system.
			usb_device_state=BASIC_USB_STATE;
			}
			powersense=0;			// running only from batteries now, ie. USB cable has been disconnected!
			lowPowerModeEnable=1;	// enable power saving!
		}
		if((autobacklightmode!=0)&&(usbConnecting==0))checkAutoBackLight();
		// an1=0 usb_device_sate> BASIC STATE implies that connected by USB cable no battery
		// an1=0 usb_device_state<=BASIC_STATE implies that connected by battery no USB
		// an1>0 means battery and USB power.
		// update the last good supply voltage if connected by USB power (either USB only or USB and battery)
		if((an1>0)||((an1==0)&&(usb_device_state>BASIC_USB_STATE)))lastGoodSupplyVoltage=(int)((float)usbRegAssumedVoltage*1023.0/(float)an0);
		if(hlvdreq!=0)
		{
			PIR2bits.HLVDIF=0;
			PIE2bits.HLVDIE=1;
			hlvdreq=0;
		}
	}
	else
	{
		// this is the extra low power bit
		if(PORTBbits.RB6==0){ serveKeyPress(0); extralowCount=0; }
		extralowCount=extralowCount % LOW_POWER_PERIOD;
		if(extralowCount==0)
		{
		displayClockMode(clockmode);
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		refreshScreen();
		extralowCount=0;
		} else
		{
		 if(extralowCount==1)
		 {
			if(an0>0)supplyVoltage=(int)((float)usbRegAssumedVoltage*1023.0/(float)an0); else supplyVoltage=0;
			if(supplyVoltage>(actualtrippoint+hysterisis))
			{
			// we must exit the extra low voltage trip point here...
			powermode=0;
			actualtrippoint=initHLVD(0, lvtripPoint);			// put it back up! note initHLVD sets powermode to zero
			}
		 }	
		 digits[0]=0;
		 digits[1]=0;
		 digits[2]=0;
		 digits[3]=0;
		 decimalPoint=0;
		 refreshScreen();
		 refreshScreen();
		}
	}
	extralowCount++;
	PIR1bits.TMR1IF=0;
	goToLowPower();
}

#pragma interruptlow myISR
void myISR(void)
{
	OSCCONbits.SCS1=0;
	OSCCONbits.SCS0=0;			// note: SCS1 should be cleared first!
	//INTCONbits.GIEL=0;		// disable any other low interrupt, to avoid stack overflows!
	if((UIR & UIE)!=0)
	{
	USBDriverService();
	BootService();
	PIR2bits.USBIF=0;
	} 
	if(PIR2bits.TMR3IF==1)
		{ 	
			refreshScreen();
			if(PORTBbits.RB6==0)serveKeyPress(BUTTON_DELAY);
			TMR3H=screenPeriod>>8;
			TMR3L=screenPeriod;
			if(!timeOut())
			{
			
			} else 
			{
				// this is the automatic dimming function! 99% hit...
				if(currentPWMDuty>0)setPWMDuty((int)(0.99*currentPWMDuty));
			}
			PIR2bits.TMR3IF=0;
		}

	if(PIE2bits.HLVDIE && PIR2bits.HLVDIF)
		{
			doHLVDInterrupt();
			PIR2bits.HLVDIF=0;
		}
	
	if(requestReset!=0)
	{
			if((requestReset & 0x02)==0)WriteEEPROM(EEPROM_WRITTEN_BASE_ADDR, MAGIC_VALUE+1);
			_asm
				reset
			_endasm
	}
	//INTCONbits.GIEL=1;		// re-enable any other low interrupt...
	goToLowPower();
}

void disAdec(int x, int cursor)
{
	int l;
	l=(x%10);
	digits[cursor]=sevenSegment[0x0F & (((x-l)/10)%10)];
	digits[cursor+1]=sevenSegment[0x0F & l];
}

void disA(int x, int cursor)
{
	// in Hex
	digits[cursor]=sevenSegment[0x0F & (x>>4)];
	digits[cursor+1]=sevenSegment[0x0F & x];
}

void disWdec(int x)
{
	int l;
	if(x>9999)x=9999; else if(x<0)x=0;
	l=(x % 10);
	digits[3]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[2]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[1]=sevenSegment[0x0F & l];
	x=x-l;
	x=x/10;
	l=(x % 10);
	digits[0]=sevenSegment[0x0F & l];
}

// these are bounds in mV
rom int hlv_bound[16]={ 2170, 2230, 2360, 2440, 2600, 2790, 2890, 3120, 3390, 3550, 3710, 3900, 4110, 4330, 4590, 10000 };

int initHLVD(unsigned char mode, int voltage_level)
{
	// mode=0 falling level (use bounds_min levels)
	// mode=1 rising level (use the bounds_max levels)
	// voltage_level = to set in mV
	// returns the actual level set in mV...
	int i, k;
	//
	PIE2bits.HLVDIE=0;		// disable Interrupt.
	HLVDCONbits.HLVDEN=0;	// disable module
	k=0;					// for default!
	for(i=0; i<16; i++)
	{
	if((voltage_level>hlv_bound[i]) && (voltage_level<=hlv_bound[i+1]))k=i;
	}
	actualtrippoint=hlv_bound[k];
	if(mode==0)HLVDCONbits.VDIRMAG=0; else HLVDCONbits.VDIRMAG=1;
	HLVDCON&=0xF0;
	HLVDCON|=(0x0F & k);
	HLVDCONbits.HLVDEN=1;
	IPR2bits.HLVDIP=0;			// low interrupt priority!
	//PIE2bits.HLVDIE=1;		// do not enable interrupt!
								// zero powermode to take into account places where you are at a powermode level say 2
	hlvdreq=1;					// and the user requests through setLVTrip (which calls this function) to set a level
	return actualtrippoint;		// higher than the current supply voltage. In that case, since we zero powermode, we are
}								// effectively resetting the finite state machine serviced by doHLVDInterrupt();
								// what should happen is that the state will go from 0 to 2 again...
void initLCD(void)
{
		int i;
		PORTA=0xFF;
		PORTB=0xFF;
		PORTC=0xFB;
		PORTD=0xFF;
		PORTE=0xFF;
		TRISA&=~0b00110100;
		TRISB&=~0b00011111;
		TRISC&=~0b11000100;			
		TRISC|=0b00110000;		// bits RC5, RC4 are always inputs. (they are USB shared pins).
		TRISD&=~0b11111100;
		TRISE&=~0b00000111;
		for(i=0; i<4; i++)digits[i]=0x40;
		decimalPoint=0;
		screenPeriod=0xFD00;
		screenCounter=0;
		screenRefreshes=0;
		T3CON=0x83;
		PIR2bits.TMR3IF=0;
		IPR2bits.TMR3IP=0;
		PIE2bits.TMR3IE=1;
}

int keyPressed(void)
{
	int i;
	i=PORTB;
	i=i & 0xC0;
	i=i >> 6;
	return i;
}

int waitForKey(void)
{
	int i;
	i=0xFF;
	while(i!=0)i=keyPressed();
	while(i==0)i=keyPressed();
	return i;
}

void initRTC(void)
{
	// real time clock using external 32768 Hz crystal.
	T1CON=0x0E;					// do not synchronise LPT1 oscillator with system clock, they are asynchronous.
	TMR1H=0x00;
	TMR1L=0x00;					// to write we write L last to read read L first for 16 bit operation.
	PIR1bits.TMR1IF=0;
	IPR1bits.TMR1IP=1;			// high interrupt
	DelayMs(100);				// to allow oscillator to stabilize
	PIE1bits.TMR1IE=1;			// enable the interrupt!
	T1CONbits.TMR1ON=1;			// turn T1 on.
}

unsigned int readAN(unsigned char channel)
{
	unsigned int i;
	ADCON0&=0xC3;
	i=0x3C & ((channel & 0x0F)<<2);
	ADCON0|=i;				// set the channel
	ADCON0|=0x02;
	while((ADCON0 & 0x02)!=0);
	i=ADRESH*256 + ADRESL;
	return i;
}

void updateANs(void)
{
	// update the an0 and an1 values
	ADCON0=0x01;			// channel AN0
	ADCON0bits.GO=1;
	while(ADCON0bits.GO==1);
	an0=(ADRESH<<8)+ADRESL;
	if(currentPWMDuty==0)
	{
	ADCON0=0x05;			// channel AN1
	ADCON0bits.GO=1;
	while(ADCON0bits.GO==1);
	an1=(ADRESH<<8)+ADRESL; 
	} 			// note an1 not accurate when backlight on, so don't update it when the backlight is on.
}

int initADC(void)
{
	ADCON1=0x0D;			// enable AN0 and AN1 as analog inputs!
	ADCON2=0xA5;			// right justified, FOSC/16 (should not be lower than this, ie faster) and 8 TAD
	ADCON0=0x01;
}

void setPWMDuty(int percentage)
{
	// f is a percentage from 0 to 100 %
	int i;
	// backlight will not turn on if in powermodes 2, 4...
	if((powermode & 0x06)!=0)percentage=0; else if(percentage<MIN_PWM_DUTY)percentage=0; else if(percentage>99)percentage=100;
	currentPWMDuty=percentage;
	i=((T2_PERIOD_DEFAULT+1)*percentage)/100;
	if(i<0)i=0; else if(i>0xFF)i=0xFF;
	CCPR1L=i;				// only 8 bits are used...
}

void hitBackLightPWMDuty(int percentage, int period)
{
	setPWMDuty(percentage);
	setTimeOut(period);
}

int initPWM(void)
{
	CCP1CON=0;
	T2CON=0;
	PR2=T2_PERIOD_DEFAULT;	// this gives a PWM frequency arounf 30kHz at 20MHz system clock
	setPWMDuty(0);
	TRISC&=~0x04;			// make RC2 an output which is multiplexed with CCP1 (pin 36 of TQFP package)
	T2CON=0x05;				// turn on Timer 2 with 1:2 pre and 1:1 post scaling
	PIR1bits.CCP1IF=0;		// clear interrupt pending
	PIE1bits.CCP1IE=0;		// disable interrupt
	IPR1bits.CCP1IP=0;		// low interrupt priority, if interrupt is enabled
	CCP1CON=0x0C;			// enable PWM mode for CCP1 module
}

void initSystem(float usbregvoltage)
{
	INTCON2bits.RBPU=0;
	usbRegAssumedVoltage=(int)(1000.0*usbregvoltage);
	setTimeOut(0);
	initADC();
	initLCD();
	initRTC();
	initPWM();
	RCONbits.IPEN=1;		// enable prioritized interrupts!
	INTCONbits.GIEL=1;		// enable low interrupts!
	INTCONbits.GIE=1;		// enable all interrupts!!
}

/*
float readVCC()
{
	// assumes that the output of the 3.3V internal regulator for USB
	// is connected via a 33k ohm resistor to the AN0 input
	float t;
	int i;
	i=readAN(0);
	if(i>0)
	{ 
		t=usbRegAssumedVoltage*1.023000/((float)i);	
	} 
	else
	{
		t=0.0;
	}		
	return t;
}*/

void wait(void)
{
	_asm
		nop
	_endasm
}

void displayVersion(void)
{
	digits[0]=0x71;			// F
	digits[1]=sevenSegment[ VER_HIGH & 0x0F ];
	disAdec(VER_LOW, 2);
	decimalPoint=4;
}

void USBConnect(void)
{
	int xx;
	char decp, dir;
	//
	usbConnecting=1;
	setPWMDuty(0);
	lowPowerModeEnable=0;
	screenUpdateOn=0;
	digits[0]=0x3E;				// U
	digits[1]=0x6D;				// S
	digits[2]=0x7C;				// B
	digits[3]=0x40;				// -
	decimalPoint=0;
	if(powersense==1)xx=USB_CONNECT_TIMEOUT; else xx=USB_CONNECT_TIMEOUT_BATTERY;
	mInitializeUSBDriver();     // See usbdrv.h
 	USBCheckBusStatus();        // Modified to always enable USB module
	setSystemTimeOut(xx);		// time allowed to connect before timeout in seconds
	decp=4;
	dir=0;
	while((usb_device_state!=CONFIGURED_STATE)&&(!systemTimeOut()))
	{
		digits[3]=sevenSegment[ usb_device_state & 0x0F];
		decimalPoint=decp;
		if(dir==0)
		{
			 if(decp==4)decp=8; else if(decp==8){ decp=4; dir^=1; }
		} else
		{
			if(decp==4)decp=2; else if(decp==2){ decp=4; dir^=1; }
		}
		DelayMs(250);	
	}
	digits[3]=sevenSegment[ usb_device_state & 0x0F];
	DelayMs(200);
	decimalPoint=0;
	screenUpdateOn=1;
	setSystemTimeOut(0);
	usbConnecting=0;
}

void setAutoBackLightMode(int y)
{
	int i;
	i=autobacklightmode;
	i&=0x80;
	if(y==1)
	{	
		i|=1;
	} else
	if(y==2)
	{
		i|=2;
	}
	autobacklightmode=i;
	WriteEEPROM(AUTOBACKLIGHTMODE_ADDR,(i & 3));
}

void setTwentyFourHourMode(int y)
{
	if(y==0)
	{
		twentyfourhourmode=0;
	} else
	if(y==1)
	{
		twentyfourhourmode=1;
	} 
	WriteEEPROM(TWENTYFOURHOURMODE_ADDR,twentyfourhourmode);
}

void setClockMode(char imode)
{
	int i;
	i=(imode % MODE_MODULUS);
	if(i!=clockmode){
		clockmode=i;
		displayCount=0;
		syncCount=0;
		loCount=0;
	}
	WriteEEPROM(CLOCKMODE_ADDR, clockmode);
}

void setTimeOutPeriod(int per)
{
	if(per<1)per=1;				// 1 is the minimum
	timeOutPeriod=per;
	WriteEEPROM(TIMEOUTPERIODHIGH_ADDR, (timeOutPeriod>>8));
	WriteEEPROM(TIMEOUTPERIODLOW_ADDR, timeOutPeriod);
}

void setLowPowerPWMDuty(int y)
{
	if(y<MIN_PWM_DUTY)y=0;
	else if(y>99)y=100;
	lowPowerPWMDuty=y;
	WriteEEPROM(LOWPOWERPWMDUTY_ADDR,lowPowerPWMDuty);
}

void setDisplayTimeOut(int y)
{
	if(y<1)y=1;					// 1 is the minimum
	displaytimeout=y;
	WriteEEPROM(DISPLAYTIMEOUTHIGH_ADDR,(displaytimeout>>8));
	WriteEEPROM(DISPLAYTIMEOUTLOW_ADDR,displaytimeout);
}

void setDefaultClockMode(char imode)
{
	scrollingmode=(imode % MODE_MODULUS);
	WriteEEPROM(SCROLLINGMODE_ADDR,scrollingmode);
}

void setSenseResistor(int y)
{
	if(y>0)senseresistor=y; else senseresistor=SENSE_RESISTOR_DEFAULT;
	WriteEEPROM(SENSERESISTORHIGH_ADDR, (senseresistor>>8));
	WriteEEPROM(SENSERESISTORLOW_ADDR, senseresistor);
}

void setVoltageRef(int y)
{
	if(y>0)usbRegAssumedVoltage=y;
	WriteEEPROM(USBREGASSUMEDVOLTAGEHIGH_ADDR, (usbRegAssumedVoltage>>8));
	WriteEEPROM(USBREGASSUMEDVOLTAGELOW_ADDR, (usbRegAssumedVoltage));
}

void setLVTrip(int y)
{
	// zero powermode to take into account places where you are at a powermode level say 2
	// and the user requests through setLVTrip (which calls this function) to set a level
	// higher than the current supply voltage. In that case, since we zero powermode, we are
	// effectively resetting the finite state machine serviced by doHLVDInterrupt();
	// what should happen is that the state will go from 0 to 2 again...
	powermode=0;
	lvtripPoint=initHLVD(0, y);
	WriteEEPROM(LVTRIPPOINTHIGH_ADDR, (lvtripPoint>>8));
	WriteEEPROM(LVTRIPPOINTLOW_ADDR, (lvtripPoint));
}

void setNoBattery(char t)
{
	if(t==1)noBattery=1; else noBattery=0;
	WriteEEPROM(NOBATTERY_ADDR, noBattery);
}

void setHysterisis(int y)
{
	if(y<0)y=0; else if(y>MAX_HYSTERISIS)y=MAX_HYSTERISIS;
	hysterisis=y;
	WriteEEPROM(HYSTERISISHI_ADDR, (hysterisis>>8));
	WriteEEPROM(HYSTERISISLO_ADDR, (hysterisis));
}

void resetDisplayTimeOut(void)
{
	displayCountDown=displaytimeout;
}

void updateAllSettings(void)
{
	setLVTrip(lvtripPoint);
	setSenseResistor(senseresistor);
	setLowPowerPWMDuty(lowPowerPWMDuty);
	setTimeOutPeriod(timeOutPeriod);
	setDisplayTimeOut(displaytimeout);
	setDefaultClockMode(scrollingmode);
	setClockMode(clockmode);
	setAutoBackLightMode(autobacklightmode);
	resetDisplayTimeOut();
	setTimeOut(0);
	setHysterisis(hysterisis);
	setNoBattery(noBattery);
}

void loadDefaults(void)
{
	hysterisis=HYSTERISIS_DEFAULT;
	lastGoodSupplyVoltage=BATTERY_HIGHEST_VOLTAGE;
	setNoBattery(NOBATTERY_DEFAULT);
	setTwentyFourHourMode(TWENTYFOURHOUR_DEFAULT);
	lvtripPoint=LOW_POWER_LV_TRIP;
	setLVTrip(lvtripPoint);
	setSenseResistor(SENSE_RESISTOR_DEFAULT);			// in mOhms
	setLowPowerPWMDuty(MAX_PWM_DUTY_LOW_POWER);
	setTimeOutPeriod(DEFAULT_TIMEOUT);
	setDisplayTimeOut(DEFAULT_DISPLAY_TIMEOUT);
	setDefaultClockMode(DEFAULT_DEFAULT_DISPLAY_MODE);
	setClockMode(DEFAULT_DISPLAY_MODE);
	setAutoBackLightMode(DEFAULT_AUTO_BACKLIGHT_MODE);
	resetDisplayTimeOut();
}

int initSequence(void)
{
	int i;
	// these are the defaults
	syncNeeded=1;
	displayCount=0;
	syncCount=0;
	loCount=0;
	extralowCount=0;
	powermode=0;
	buttonkey=0;
	buttonpressedlast=0;
	buttonpressednow=0;
	scrollingmode=0;
	an0=800;
	an1=1;
	screenUpdateOn=0;
	RTCseconds=0;
	RTCminutes=0;
	RTChours=12;
	RTCwday=2;					// 0=Sunday 2=Tuesday
	RTCmday=1;					// 1st of January 2008 was a Tuesday.			
	RTCyday=0;
	RTCisdst=0;
	RTCmonth=0;
	RTCyear=2008-YEAR_OFFSET;
	lastseconds=RTCseconds;
	lastminutes=RTCminutes;
	lasthours=RTChours;
	lastwday=RTCwday;
	lastmday=RTCmday;
	lastyday=RTCyday;
	lastisdst=RTCisdst;
	lastyear=RTCyear;
	lastmonth=RTCmonth;
	verhigh=VER_HIGH;
	verlow=VER_LOW;
	//
	lowPowerModeEnable=0;
	IPR2bits.USBIP=0;			// select low interrupt priority (added MG)
	UIR=0;
	PIR2bits.USBIF=0;
	PIE2bits.USBIE=1;			// enable USB interrupts!	(added MG)
	lowPowerModeEnable=0;
	initSystem(VOLTAGE_REF_DEFAULT/1000.0);
	restoreAllSettings();
	updateANs();
	if((noBattery==1)||(an1!=0))powersense=1; else powersense=0;		// update powersense for the first time.
	//
	USBConnect();
	displayVersion();
	DelayMs(VERSION_DELAY);
	//
	lowPowerModeEnable=1;
}

void saveAllSettings(void)
{
#if (PERSISTENT_SETTINGS_ENABLE==1)
	WriteEEPROM(CLOCKMODE_ADDR, clockmode);
	WriteEEPROM(LOWPOWERPWMDUTY_ADDR, lowPowerPWMDuty);
	WriteEEPROM(TIMEOUTPERIODHIGH_ADDR, (timeOutPeriod>>8));
	WriteEEPROM(TIMEOUTPERIODLOW_ADDR, timeOutPeriod);
	WriteEEPROM(LVTRIPPOINTHIGH_ADDR, (lvtripPoint>>8));
	WriteEEPROM(LVTRIPPOINTLOW_ADDR, lvtripPoint);
	WriteEEPROM(USBREGASSUMEDVOLTAGEHIGH_ADDR, (usbRegAssumedVoltage>>8));
	WriteEEPROM(USBREGASSUMEDVOLTAGELOW_ADDR, usbRegAssumedVoltage);
	WriteEEPROM(SENSERESISTORHIGH_ADDR, (senseresistor>>8));
	WriteEEPROM(SENSERESISTORLOW_ADDR, senseresistor);
	WriteEEPROM(SCROLLINGMODE_ADDR, scrollingmode);
	WriteEEPROM(DISPLAYTIMEOUTHIGH_ADDR, (displaytimeout>>8));
	WriteEEPROM(DISPLAYTIMEOUTLOW_ADDR, displaytimeout);
	WriteEEPROM(AUTOBACKLIGHTMODE_ADDR,(3 & autobacklightmode));
	WriteEEPROM(TWENTYFOURHOURMODE_ADDR,twentyfourhourmode);
	WriteEEPROM(NOBATTERY_ADDR, noBattery);
	WriteEEPROM(HYSTERISISHI_ADDR, (hysterisis>>8));
	WriteEEPROM(HYSTERISISLO_ADDR, (hysterisis));
	WriteEEPROM(EEPROM_WRITTEN_BASE_ADDR, MAGIC_VALUE);
#endif
}

void restoreAllSettings(void)
{
	// restore all settings from EEPROM
#if (PERSISTENT_SETTINGS_ENABLE==1)
	int i;
	i=ReadEEPROM(EEPROM_WRITTEN_BASE_ADDR);
	if(i==MAGIC_VALUE)
	{
	clockmode=ReadEEPROM(CLOCKMODE_ADDR);
	lowPowerPWMDuty=ReadEEPROM(LOWPOWERPWMDUTY_ADDR);
	timeOutPeriod=(ReadEEPROM(TIMEOUTPERIODHIGH_ADDR)<<8)+ReadEEPROM(TIMEOUTPERIODLOW_ADDR);
	lvtripPoint=(ReadEEPROM(LVTRIPPOINTHIGH_ADDR)<<8)+ReadEEPROM(LVTRIPPOINTLOW_ADDR);
	usbRegAssumedVoltage=(ReadEEPROM(USBREGASSUMEDVOLTAGEHIGH_ADDR)<<8)+ReadEEPROM(USBREGASSUMEDVOLTAGELOW_ADDR);
	senseresistor=(ReadEEPROM(SENSERESISTORHIGH_ADDR)<<8)+ReadEEPROM(SENSERESISTORLOW_ADDR);
	scrollingmode=ReadEEPROM(SCROLLINGMODE_ADDR);
	displaytimeout=(ReadEEPROM(DISPLAYTIMEOUTHIGH_ADDR)<<8)+ReadEEPROM(DISPLAYTIMEOUTLOW_ADDR);
	autobacklightmode=(3 & ReadEEPROM(AUTOBACKLIGHTMODE_ADDR));
	twentyfourhourmode=ReadEEPROM(TWENTYFOURHOURMODE_ADDR);
	noBattery=ReadEEPROM(NOBATTERY_ADDR);
	hysterisis=(ReadEEPROM(HYSTERISISHI_ADDR)<<8)+(ReadEEPROM(HYSTERISISLO_ADDR));
	updateAllSettings();
	} else
	{
		loadDefaults();
		saveAllSettings();
	}
	// done!
#else
	loadDefaults();
#endif	
}

void main(void)
{
	if((requestReset!=0)||(RCONbits.POR==0))
	{
	actualtrippoint=120;
	RCONbits.POR=1;
	requestReset=0;
	hlvdset=0;
	hlvdreq=0;
	initSequence(); 
	}
	while(1)
	{

	}
}

#pragma code user = RM_RESET_VECTOR

/** EOF main.c ***************************************************************/
